/* ****************************************************************************
 * Copyright: 2017-2025 RAYLASE GmbH
 * This source code is the proprietary confidential property of RAYLASE GmbH.
 * Reproduction, publication, or any form of distribution to
 * any party other than the licensee is strictly prohibited.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#pragma once

#include "ToString.h"

#include <chrono>
#include <fstream>
#include <mutex>
#include <sstream>
#include <string>
#include <thread>

class RpcLogger
{
private:
	static const char padding[];
	mutable std::mutex mutex;
	std::ofstream fileStream;

	std::int32_t _truncateArgumentLength = -1;

	void getTimestamp(char* outTime, std::uint32_t len, bool shortFormat = false);

	void _add() {}
	template<typename First, typename... Rest> void _add(const First& first, const Rest&... rest)
	{
		using namespace std;
		std::string tmp = to_string(first);
		if (_truncateArgumentLength >= 0 && _truncateArgumentLength < (std::int32_t)tmp.size())
			fileStream << ' ' << tmp.substr(0, _truncateArgumentLength) << "...[truncated]";
		else
			fileStream << ' ' << tmp;

		if constexpr (sizeof...(rest) != 0)
			fileStream << ',';
		_add(rest...);
	}

	void _write(const char* message);
	void _write(const char* type, const std::string& name, bool function);

public:
	RpcLogger();
	~RpcLogger();
	void Enable(const std::string& path, bool timestamp, bool append, std::int32_t truncateArgumentsAfter);
	void Disable() noexcept;
	void Write(const char* message);
	void Write(const char* type, const std::string& name, bool function = true);
	void Write(const std::string& message);
	void BeginLine(const char* type, const std::string& name);
	void EndLine();
	void Flush();

	template<typename... Args> void Append(const Args&... args)
	{
		if constexpr (sizeof...(args) != 0)
		{
			std::lock_guard<std::mutex> lock(mutex);

			try
			{
				_add(args...);
			}
			catch (...)
			{
				fileStream << "Error creating string!" << std::endl;
			}
		}
	}
};
template<> void RpcLogger::Append(const std::string& args);
